
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; MODULE      : giac-menus.scm
;; DESCRIPTION : Giac menus
;; COPYRIGHT   : (C) 1999  Bernard Parisse and Joris van der Hoeven
;;                   2021  Luka Marohnić
;;
;; This software falls under the GNU general public license version 3 or later.
;; It comes WITHOUT ANY WARRANTY WHATSOEVER. For details, see the file LICENSE
;; in the root directory or <http://www.gnu.org/licenses/gpl-3.0.html>.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(texmacs-module (giac-menus)
  (:use (utils plugins plugin-cmd)
	(doc help-funcs)
	(dynamic scripts-edit)
        (dynamic session-menu)
	(convert tools tmconcat)))

(define (giac-spaces? t)
  (and (string? t) (in? t (list "" " " "  " "   " "    "))))

(define (giac-output-simplify t)
  ;;(display* "Simplify " t "\n")
  (cond ((and (func? t 'concat) (== (length t) 4) (func? (cadr t) 'htab) (func? (caddr t) 'image))
         (plugin-output-std-simplify "giac" (caddr t)))
        ((and (func? t 'equation*) (func? (cAr t) 'document))
	 `(math ,(plugin-output-std-simplify "giac" (cAr t))))
	((match? t '(with "math-display" "true" :%1))
	 (giac-output-simplify (cAr t)))
	((match? t '(with "mode" "math" "math-display" "true" :%1))
	 `(math ,(giac-output-simplify (cAr t))))
	((func? t 'text 1)
	 `(text ,(giac-output-simplify (cAr t))))
	((func? t 'math 1)
	 `(math ,(giac-output-simplify (cAr t))))
	((func? t 'with 1)
	 (giac-output-simplify (cAr t)))
	((func? t 'with)
	 (rcons (cDr t) (giac-output-simplify (cAr t))))
        ((and (func? t 'concat) (pair? (cdr t)) (giac-spaces? (cadr t)))
         (giac-output-simplify (cons (car t) (cddr t))))
	((func? t 'concat)
	 (apply tmconcat (map giac-output-simplify (cdr t))))
	(else (plugin-output-std-simplify "giac" t))))

(define (giac-contains-output? t)
  (cond ((or (func? t 'equation*) (func? t 'image)) #t)
	((or (func? t 'concat) (func? t 'document))
	 (list-or (map giac-contains-output? (cdr t))))
	((and (func? t 'with) (nnull? (cdr t)))
	 (giac-contains-output? (cAr t)))
	((or (func? t 'text 1) (func? t 'math 1))
	 (giac-contains-output? (cAr t)))
	(else #f)))

(tm-define (plugin-output-simplify name t)
  (:require (== name "giac"))
  ;;(display* "Simplify output " t "\n")
  (display "Plugin output simplify:\n")
  (display t)
  (display "\n")
  (if (func? t 'document)
      (with u (list-find (cdr t) giac-contains-output?)
	(if u (giac-output-simplify u) ""))
      (giac-output-simplify t)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Insert Giac primitive
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (giac-cursor-pos l)
  (cond ((null? l) 0)
	((null? (cdr l)) 1)
	((and (== (car l) #\() (== (cadr l) #\))) 1)
	((and (== (car l) #\() (== (cadr l) #\,)) 1)
	((and (== (car l) #\,) (== (cadr l) #\))) 1)
	((and (== (car l) #\,) (== (cadr l) #\,)) 1)
	(else (+ (giac-cursor-pos (cdr l)) 1))))

(define (giac-insert s)
  (insert-go-to s (list (giac-cursor-pos (string->list s)))))
  
(define (giac-apply fun . opts)
  (if (not-in-session?)
      (script-apply fun (cadr opts))
      (giac-insert (string-append fun (car opts)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Giac menu
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(menu-bind giac-functions-menu
  (if (not-in-session?)
      (link scripts-eval-menu)
      ---)
  (-> "Simplification"
      ("Normalize" (giac-apply "normal" "()" 1))
      ("Reduce rational" (giac-apply "ratnormal" "()" 1))
      ("Factorize" (giac-apply "factor" "()" 1))
      ("Decompose to partial fractions" (giac-apply "partfrac" "()" 1))
      ("Simplify" (giac-apply "simplify" "()" 1))
      ("Trigonometric simplification" (giac-apply "trigsimplify" "()" 1))
      ("Reduce the number of non-rational parameters" (giac-apply "tsimplify" "()" 1)))
  (-> "Solving"
      ("Solve equations/inequalities" (giac-apply "solve" "()" 1))
      ("Solve linear equations" (giac-apply "linsolve" "()" 2))
      ("Solve recurrences" (giac-apply "rsolve" "()" 3))
      ("Solve complex equations" (giac-apply "csolve" "()" 2))
      ("Solve ODE" (giac-apply "desolve" "()" 2))
      ("Solve Diophantine equation" (giac-apply "isolve" "()" 1)))
  (-> "Plotting"
      ("Function plot" (giac-apply "plotfunc" "()" 1))
      ("Parametric function plot" (giac-apply "plotparam" "()" 1))
      ("Polar plot" (giac-apply "plotpolar" "()" 1))
      ("Implicit curve plot" (giac-apply "implicitplot" "()" 1))
      ("Plot setup" (giac-apply "xyztrange" "()" 1)))
  ---
  (-> "Constants"
      ("Euler's number" (giac-insert "e"))
      ("Imaginary unit" (giac-insert "i"))
      ("Pi" (giac-insert "pi"))
      ("Euler-Mascheroni constant" (giac-insert "euler_gamma"))
      ("Positive infinity" (giac-insert "+infinity"))
      ("Negative infinity" (giac-insert "-infinity"))
      ("Unsigned infinity" (giac-insert "infinity"))
      ("Undefined symbol" (giac-insert "undef")))
  (-> "Physical units"
      (-> "Mass"
          ("Atomic mass unit" (giac-insert "_u"))
          ("Gram" (giac-insert "_g"))
          ("Kilogram" (giac-insert "_kg"))
          ("Metric ton" (giac-insert "_t"))
          ("Short ton" (giac-insert "_ton"))
          ("Long (UK) ton" (giac-insert "_tonUK"))
          ("Grain" (giac-insert "_grain"))
          ("Carat" (giac-insert "_ct"))
          ("Pound" (giac-insert "_lb"))
          ("Ounce" (giac-insert "_oz"))
          ("UK fluid ounce" (giac-insert "_ozUK"))
          ("US fluid ounce" (giac-insert "_ozfl"))
          ("Troy pound" (giac-insert "_lbt"))
          ("Troy ounce" (giac-insert "_ozt"))
          ("Slug" (giac-insert "_slug"))
      )
      (-> "Length"
          ("Meter" (giac-insert "_m"))
          ("Centimeter" (giac-insert "_cm"))
          ("Millimeter" (giac-insert "_mm"))
          ("Kilometer" (giac-insert "_km"))
          ("Inch" (giac-insert "_in"))
          ("Mil" (giac-insert "_mil"))
          ("Rod" (giac-insert "_rod"))
          ("International foot" (giac-insert "_ft"))
          ("Survey foot" (giac-insert "_ftUS"))
          ("International yard" (giac-insert "_yd"))
          ("International mile" (giac-insert "_mile"))
          ("US statute mile" (giac-insert "_miUS"))
          ("Nautical mile" (giac-insert "_mille"))
          ("Angstrom" (giac-insert "_Angstrom"))
          ("Astronomical unit" (giac-insert "_au"))
          ("Chain" (giac-insert "_chain"))
          ("Fathom" (giac-insert "_fath"))
          ("Board foot" (giac-insert "_fbm"))
          ("Fermi" (giac-insert "_fermi"))
          ("Light year" (giac-insert "_lyr"))
          ("Parsec" (giac-insert "_pc"))
      )
      (-> "Area"
          ("Are" (giac-insert "_a"))
          ("Hectare" (giac-insert "_ha"))
          ("Acre" (giac-insert "_acre"))
          ("Barn" (giac-insert "_b"))
          ("Square international mile" (giac-insert "_mi^2"))
      )
      (-> "Volume"
          ("Liter" (giac-insert "_l"))
          ("Milliliter" (giac-insert "_ml"))
          ("Barrel" (giac-insert "_bbl"))
          ("Bushel" (giac-insert "_bu"))
          ("US bushel" (giac-insert "_buUS"))
          ("US cup" (giac-insert "_cu"))
          ("Canadian gallon" (giac-insert "_galC"))
          ("UK gallon" (giac-insert "_galUK"))
          ("US gallon" (giac-insert "_galUS"))
          ("US liquid pint" (giac-insert "_liqpt"))
          ("UK pint" (giac-insert "_ptUK"))
          ("US peck" (giac-insert "_pk"))
          ("Quart" (giac-insert "_qt"))
          ("Stere" (giac-insert "_st"))
          ("Tablespoon" (giac-insert "_tbsp"))
          ("Teaspoon" (giac-insert "_tsp"))
      )
      (-> "Angle"
          ("Radian" (giac-insert "_rad"))
          ("Tour (2π radians)" (giac-insert "_tr"))
          ("Steradian" (giac-insert "_sr"))
          ("Degree" (giac-insert "_deg"))
          ("Minute of arc" (giac-insert "_arcmin"))
          ("Second of arc" (giac-insert "_arcs"))
          ("Grade" (giac-insert "_grad"))
      )
      (-> "Velocity"
          ("Kilometers per hour" (giac-insert "_kph"))
          ("Miles per hour" (giac-insert "_mph"))
          ("Nautical miles per hour" (giac-insert "_knot"))
          ("Revolutions per minute" (giac-insert "_rpm"))
      )
      (-> "Acceleration"
          ("Meters per second squared" (giac-insert "_m*_s^-2"))
          ("Gal" (giac-insert "_Gal"))
          ("Standard freefall" (giac-insert "_ga"))
      )
      (-> "Force, weight"
          ("Newton" (giac-insert "_N"))
          ("Dyne" (giac-insert "_dyn"))
          ("Poundal" (giac-insert "_pdl"))
          ("Gram-force" (giac-insert "_gf"))
          ("Pound-force" (giac-insert "_lbf"))
          ("Kilopound-force" (giac-insert "_kip"))
      )
      (-> "Pressure"
          ("Pascal" (giac-insert "_Pa"))
          ("Atmosphere" (giac-insert "_atm"))
          ("Bar" (giac-insert "_bar"))
          ("Inches of water (60°F)" (giac-insert "_inH2O"))
          ("Inches of mercury (0°C)" (giac-insert "_inHg"))
          ("Millimeters of mercury (0°C)" (giac-insert "_mmHg"))
          ("Pounds per square inch" (giac-insert "_psi"))
      )
      (-> "Energy, work, heat"
          ("British thermal unit (BTU)" (giac-insert "_Btu"))
          ("Joule" (giac-insert "_J"))
          ("Erg" (giac-insert "_erg"))
          ("Calorie" (giac-insert "_cal"))
          ("Kilocalorie" (giac-insert "_Kcal"))
          ("Electronvolt" (giac-insert "_eV"))
          ("Megaelectronvolt" (giac-insert "_MeV"))
          ("Watt-hour" (giac-insert "_Wh"))
          ("Kilowatt-hour" (giac-insert "_kWh"))
          ("Ton of coal equivalent" (giac-insert "_tec"))
          ("Ton of oil equivalent" (giac-insert "_tep"))
          ("Liter of oil equivalent" (giac-insert "_lep"))
          ("Barrel of oil equivalent" (giac-insert "_bblep"))
          ("EEC therm" (giac-insert "_therm"))
      )
      (-> "Power"
          ("Watt" (giac-insert "_W"))
          ("Megawatt" (giac-insert "_MW"))
          ("Horsepower" (giac-insert "_hp"))
      )
      (-> "Temperature"
          ("Kelvin" (giac-insert "_K"))
          ("Rankine" (giac-insert "_Rankine"))
          ("Degree Fahrenheit" (giac-insert "_degreeF"))
      )
      (-> "Radiation"
          ("Becquerel" (giac-insert "_Bq"))
          ("Curie" (giac-insert "_Ci"))
          ("Gray" (giac-insert "_Gy"))
          ("Roentgen" (giac-insert "_R"))
          ("Sievert" (giac-insert "_Sv"))
          ("Rad" (giac-insert "_rd"))
          ("Rem" (giac-insert "_rem"))
      )
      (-> "Electricity"
          ("Ampere" (giac-insert "_A"))
          ("Farad" (giac-insert "_F"))
          ("Faraday" (giac-insert "_Fdy"))
          ("Henry" (giac-insert "_H"))
          ("Ohm" (giac-insert "_Ohm"))
          ("Siemens" (giac-insert "_S"))
          ("Volt" (giac-insert "_V"))
      )
      (-> "Magnetic field"
          ("Tesla" (giac-insert "_T"))
          ("Weber" (giac-insert "_Wb"))
      )
      (-> "Time"
          ("Second" (giac-insert "_s"))
          ("Minute" (giac-insert "_mn"))
          ("Hour" (giac-insert "_h"))
          ("Day" (giac-insert "_d"))
          ("Year" (giac-insert "_yr"))
      )
      (-> "Frequency"
          ("Hertz" (giac-insert "_Hz"))
          ("Megahertz" (giac-insert "_MHz"))
      )
      (-> "Viscosity"
          ("Stokes" (giac-insert "_St"))
          ("Poise" (giac-insert "_P"))
      )
      (-> "Illuminance"
          ("Lumen" (giac-insert "_lm"))
          ("Lux" (giac-insert "_lx"))
          ("Lambert" (giac-insert "_lam"))
          ("Candela" (giac-insert "_cd"))
          ("Footcandle" (giac-insert "_fc"))
          ("Footlambert" (giac-insert "_flam"))
          ("Phot" (giac-insert "_ph"))
          ("Stilb" (giac-insert "_sb"))
      )
      (-> "Substance amount"
          ("Mole" (giac-insert "_mol"))
          ("Gram-mole" (giac-insert "_gmol"))
          ("Pound-mole" (giac-insert "_lbmol"))
      )
      (-> "Other"
          ("Decibel" (giac-insert "_dB"))
          ("Tex" (giac-insert "_tex"))
          ("Unit conversion" (giac-apply "mksa" "()" 2))
      ))
  (-> "Physical constants"
      ("Faraday constant" (giac-insert "_F_"))
      ("Gravitational constant" (giac-insert "_G_"))
      ("Reference intensity" (giac-insert "_I0_"))
      ("Avogadro's number" (giac-insert "_NA_"))
      ("Power at the surface of the Sun" (giac-insert "_PSun_"))
      ("Radius of the Earth" (giac-insert "_REarth_"))
      ("Radius of the Sun" (giac-insert "_RSun_"))
      ("Universal gas constant" (giac-insert "_R_"))
      ("Rydberg constant" (giac-insert "_Rinfinity_"))
      ("Standard pressure" (giac-insert "_StdP_"))
      ("Standard temperature" (giac-insert "_StdT_"))
      ("Molar volume" (giac-insert "_Vm_"))
      ("Bohr radius" (giac-insert "_a0_"))
      ("Fine structure constant" (giac-insert "_alpha_"))
      ("Wien displacement constant" (giac-insert "_c3_"))
      ("Speed of light in vacuum" (giac-insert "_c_"))
      ("Permittivity of vacuum" (giac-insert "_epsilon0_"))
      ("Dielectric constant of Silicon dioxide" (giac-insert "_epsilonox_"))
      ("Dielectric constant" (giac-insert "_epsilonsi_"))
      ("Photon frequency" (giac-insert "_f0_"))
      ("Acceleration of gravity" (giac-insert "_g_"))
      ("Planck constant" (giac-insert "_h_"))
      ("Dirac constant" (giac-insert "_hbar_"))
      ("Boltzmann constant" (giac-insert "_k_"))
      ("Photon wavelength" (giac-insert "_lambda0_"))
      ("Mass of the Earth" (giac-insert "_mEarth_"))
      ("Electron rest mass" (giac-insert "_me_"))
      ("Electron charge" (giac-insert "_q_"))
      ("Proton rest mass" (giac-insert "_mp_"))
      ("Permeability of vacuum" (giac-insert "_mu0_"))
      ("Bohr magneton" (giac-insert "_muB_"))
      ("Nuclear magneton" (giac-insert "_muN_"))
      ("Magnetic flux quantum" (giac-insert "_phi_"))
      ("Duration of a sidereal day" (giac-insert "_sd_"))
      ("Duration of a sidereal year" (giac-insert "_syr_"))
      ("Stefan-Boltzmann constant" (giac-insert "_sigma_")))
  ---
  (-> "Real numbers"
      ("Absolute value" (giac-apply "abs" "()" 1))
      ("Maximal element" (giac-apply "max" "()" 1))
      ("Minimal element" (giac-apply "min" "()" 1))
      ("Sign function" (giac-apply "sign" "()" 1)))
  (-> "Complex numbers"
      ("Absolute value" (giac-apply "abs" "()" 1))
      ("Argument" (giac-apply "arg" "()" 1))
      ("Conjugate" (giac-apply "conj" "()" 1))
      ("Imaginary part" (giac-apply "im" "()" 1))
      ("Real part" (giac-apply "re" "()" 1)))
  (-> "Elementary functions"
      (-> "Exponential and logarithmic functions"
          ("Exponential function" (giac-apply "exp" "()" 1))
          ("Natural logarithm" (giac-apply "ln" "()" 1))
          ("Common logarithm (base 10)" (giac-apply "log10" "()" 1)))
      (-> "Trigonometric and cyclometric functions"
          ("Sine" (giac-apply "sin" "()" 1))
          ("Cosine" (giac-apply "cos" "()" 1))
          ("Tangent" (giac-apply "tan" "()" 1))
          ("Cotangent" (giac-apply "cot" "()" 1))
          ("Arc sine" (giac-apply "asin" "()" 1))
          ("Arc cosine" (giac-apply "acos" "()" 1))
          ("Arc tangent" (giac-apply "atan")))
      (-> "Hyperbolic functions"
          ("Hyperbolic sine" (giac-apply "sinh" "()" 1))
          ("Hyperbolic cosine" (giac-apply "cosh" "()" 1))
          ("Hyperbolic tangent" (giac-apply "tanh" "()" 1))
          ("Inverse hyperbolic sine" (giac-apply "asinh" "()" 1))
          ("Inverse hyperbolic cosine" (giac-apply "acosh" "()" 1))
          ("Inverse hyperbolic tangent" (giac-apply "atanh" "()" 1))))
  (-> "Special functions"
      ("Ai (Airy function)" (giac-apply "Airy_Ai" "()" 1))
      ("Bi (Airy function)" (giac-apply "Airy_Bi" "()" 1))
      ("Gamma" (giac-apply "Gamma" "()" 1))
      ("Lambert W" (giac-apply "LambertW" "()" 1))
      ("Psi" (giac-apply "Psi" "()" 1))
      ("Zeta" (giac-apply "Zeta" "()" 1)))
  (-> "Rewriting"
      (-> "Exponentials and logarithms"
          ("Hyperbolic to exponential" (giac-apply "hyp2exp" "()" 1))
          ("Linearize exponentials" (giac-apply "lin" "()" 1))
          ("Collect logarithms" (giac-apply "lncollect" "()" 1))
          ("Complex exponential to sine and cosine" (giac-apply "sincos" "()" 1))
          ("Trigonometric expression to exponential" (giac-apply "trig2exp" "()" 1))
          ("Lower the number of non-rational variables" (giac-apply "tsimplify" "()" 1)))
      (-> "Trigonometric"
          ("Half-tangent" (giac-apply "halftan" "()" 1))
          ("Tangent to sine/cosine" (giac-apply "tan2sincos" "()" 1))
          ("Tangent to sine/cosine of double argument" (giac-apply "tan2sincos2" "()" 1))
          ("Collect transcendental expressions" (giac-apply "tcollect" "()" 1))
          ("Expand transcendental expressions" (giac-apply "texpand" "()" 1))
          ("Simplify by privileging sine" (giac-apply "trigsin" "()" 1))
          ("Simplify by privileging cosine" (giac-apply "trigcos" "()" 1))
          ("Simplify by privileging tangent" (giac-apply "trigtan" "()" 1))
          ("Trigonometric linearization" (giac-apply "tlin" "()" 1)))
      (-> "Inverse trigonometric rewriting"
          ("Arc cosine to arc sine" (giac-apply "acos2asin" "()" 1))
          ("Arc cosine to arc tangent" (giac-apply "acos2atan" "()" 1))
          ("Arc sine to arc cosine" (giac-apply "asin2acos" "()" 1))
          ("Arc sine to arc tangent" (giac-apply "asin2atan" "()" 1))
          ("Arc tangent to arc cosine" (giac-apply "atan2acos" "()" 1))
          ("Arc tangent to arc sine" (giac-apply "atan2asin" "()" 1))))
  ---
  (-> "Integer arithmetic"
      ("Greatest common divisor" (giac-apply "gcd" "()" 1))
      ("Largest common multiple" (giac-apply "lcm" "()" 1))
      ("Euler totient function" (giac-apply "euler" "()" 1))
      ("Chinese remainders" (giac-apply "ichinrem" "()" 1))
      ("Extended greatest common divisor" (giac-apply "iegcd" "()" 1))
      ("Integer factorization" (giac-apply "ifactor" "()" 1))
      ("Prime factors" (giac-apply "ifactors" "()" 1))
      ("Quotient" (giac-apply "iquo" "()" 1))
      ("Remainder" (giac-apply "irem" "()" 1))
      ("Prime number testing" (giac-apply "is_prime" "()" 1))
      ("Next prime" (giac-apply "nextprime" "()" 1))
      ("Previous prime" (giac-apply "prevprime" "()" 1)))
  (-> "Polynomial arithmetic"
      ("Greatest common divisor" (giac-apply "gcd" "()" 1))
      ("Largest common multiple" (giac-apply "lcm" "()" 1))
      ("Chinese remainders" (giac-apply "chinrem" "()" 1))
      ("Cyclotomic" (giac-apply "cyclotomic" "()" 1))
      ("Divisors of a polynomial" (giac-apply "divis" "()" 1))
      ("Extended greatest common divisor" (giac-apply "egcd" "()" 1))
      ("Factorization" (giac-apply "factor" "()" 1))
      ("Coefficients from roots" (giac-apply "pcoeff" "()" 1))
      ("Evaluation from coefficients" (giac-apply "peval" "()" 1))
      ("Roots from coefficients" (giac-apply "proot" "()" 1))
      ("Quotient" (giac-apply "quo" "()" 1))
      ("Remainder" (giac-apply "rem" "()" 1))
      ("Quotient and remainder" (giac-apply "quorem" "()" 1))
      ("Coefficients" (giac-apply "e2r" "()" 1))
      ("Polynomial from coefficients" (giac-apply "r2e" "()" 1))
      (-> "Orthogonal polynomials"
          ("Hermite polynomial" (giac-apply "hermite" "()" 1))
          ("Laguerre polynomial" (giac-apply "laguerre" "()" 1))
          ("Legendre polynomial" (giac-apply "legendre" "()" 1))
          ("Tchebyshev polynomial of the first kind" (giac-apply "tchebyshev1" "()" 1))
          ("Tchebyshev polynomial of the second kind" (giac-apply "tchebyshev2" "()" 1))))
  (-> "Interpolation"
      ("Lagrange polynomial" (giac-apply "lagrange" "()" 1))
      ("Spline interpolation" (giac-apply "spline" "()" 1))
      ("Rational interpolation" (giac-apply "ratinterp" "()" 1))
      ("Trigonometric interpolation" (giac-apply "triginterp" "()" 1)))
  (-> "Curve fitting"
      ("Linear regression" (giac-apply "linear_regression" "()" 1))
      ("Polynomial regression" (giac-apply "polynomial_regression" "()" 1))
      ("Exponential regression" (giac-apply "exponential_regression" "()" 1))
      ("Logarithmic regression" (giac-apply "logarithmic_regression" "()" 1))
      ("Logistic regression" (giac-apply "logistic_regression" "()" 1))
      ("Fit polynomial to data" (giac-apply "fitpoly" "()" 1)))
  (-> "Function approximation"
      ("L2-norm polynomial approximation" (giac-apply "fitpoly" "()" 1))
      ("Minimax polynomial approximation" (giac-apply "minimax" "()" 1)))
  ---
  (-> "Linear algebra"
      (-> "Vectors"
          ("Absolute value" (giac-apply "abs" "()" 1))
          ("Norm" (giac-apply "norm" "()" 1))
          ("L1 norm" (giac-apply "l1norm" "()" 1))
          ("L2 norm" (giac-apply "l2norm" "()" 1))
          ("Scalar product" (giac-apply "dot" "()" 1))
          ("Vector product" (giac-apply "cross" "()" 1)))
      (-> "Matrices"
          ("Hadamard" (giac-apply "hadamard" "()" 1))
          ("Inverse" (giac-apply "inv" "()" 1))
          ("Trace" (giac-apply "trace" "()" 1))
          ("Transconjugate" (giac-apply "trn" "()" 1))
          ("Transpose" (giac-apply "tran" "()" 1))
          ("Norm" (giac-apply "norm" "()" 1))
          ---
          ("Identity" (giac-apply "idn" "()" 1))
          ("Hilbert" (giac-apply "hilbert" "()" 1))
          ("Vandermonde" (giac-apply "vandermonde" "()" 1))
          ("Random" (giac-apply "ranm" "()" 1)))
      (-> "Eigenvalues and eigenvectors"
          ("Eigenvectors" (giac-apply "egv" "()" 1))
          ("Eigenvalues" (giac-apply "egvl" "()" 1))
          ("Jordan form" (giac-apply "jordan" "()" 1)))
      (-> "Matrix factorization"
          ("Cholesky decomposition" (giac-apply "cholesky" "()" 1))
          ("LU decomposition" (giac-apply "lu" "()" 1))
          ("QR decomposition" (giac-apply "qr" "()" 1))
          ("Schur decomposition" (giac-apply "schur" "()" 1))
          ("Singular value decomposition" (giac-apply "svd" "()" 1)))
      (-> "Gauss-Jordan pivot"
          ("Basis" (giac-apply "basis" "()" 1))
          ("Basis intersection" (giac-apply "ibasis" "()" 1))
          ("Kernel" (giac-apply "ker" "()" 1))
          ("Image" (giac-apply "image" "()" 1))
          ("Row reduction to echelon form" (giac-apply "rref" "()" 1)))
      (-> "Isometries"
          ("Elements of a 2D or 3D isometry" (giac-apply "isom" "()" 1))
          ("Isometry matrix" (giac-apply "mkisom()"))))
  (-> "Calculus"
      ("Limit" (giac-apply "limit" "()" 2))
      ("Differentiate" (giac-apply "diff" "()" 1))
      ("Integrate" (giac-apply "integrate" "()" 1))
      ("Gradient" (giac-apply "grad" "()" 1))
      ("Divergence" (giac-apply "divergence" "()" 1))
      ("Curl" (giac-apply "curl" "()" 1))
      ("Hessian" (giac-apply "hessian" "()" 1))
      ("Laplacian" (giac-apply "laplacian" "()" 1))
      ("Local extrema" (giac-apply "extrema" "()" 1))
      ("Test convexity" (giac-apply "convex" "()" 1))
      ("Implicit differentiation" (giac-apply "implicitdiff" "()" 1))
      ("Taylor expansion" (giac-apply "taylor" "()" 1))
      ("Series expansion" (giac-apply "series" "()" 1)))
  (-> "Calculus of variations"
      ("Euler-Lagrange equations" (giac-apply "euler_lagrange" "()" 1))
      ("Jacobi equation" (giac-apply "jacobi_equation" "()" 1))
      ("Conjugate equation" (giac-apply "conjugate_equation" "()" 1)))
  (-> "Numeric analysis"
      ("Newton method" (giac-apply "newton" "()" 1))
      ("Solve equations" (giac-apply "fsolve" "()" 2))
      ("Solve differential equations" (giac-apply "odesolve" "()" 3))
      ("Solve boundary-value problems" (giac-apply "bvpsolve" "()" 3))
      ("Numerical integration" (giac-apply "romberg" "()" 4))
      ("Numerical differentiation" (giac-apply "numdiff" "()" 3)))
  ---
  (-> "Probability and statistics"
      (-> "Statistics"
          ("Mean" (giac-apply "mean" "()" 1))
          ("Variance" (giac-apply "variance" "()" 1))
          ("Standard deviation" (giac-apply "stddev" "()" 1))
          ("Population standard deviation" (giac-apply "stdDev" "()" 1))
          ("Median" (giac-apply "median" "()" 1))
          ("Quartiles" (giac-apply "quartiles" "()" 1))
          ("Quantiles" (giac-apply "quantiles" "()" 2))
          ("Classes" (giac-apply "classes" "()" 1))
          ("Frequencies" (giac-apply "frequencies" "()" 1))
          ("Cumulative frequencies" (giac-apply "cumulated_frequencies" "()" 1))
          ("Covariance" (giac-apply "covariance" "()" 1))
          ("Correlation" (giac-apply "correlation" "()" 1))
          (-> "Plotting"
              (-> "Regression plot"
                  ("Linear" (giac-apply "linear_regression_plot" "()" 1))
                  ("Polynomial" (giac-apply "polynomial_regression_plot" "()" 1))
                  ("Exponential" (giac-apply "exponential_regression_plot" "()" 1))
                  ("Logarithmic" (giac-apply "logarithmic_regression_plot" "()" 1))
                  ("Logistic" (giac-apply "logistic_regression_plot" "()" 1)))
              ("Sequence plot" (giac-apply "plotseq" "()" 1))
              ("Polygonal path" (giac-apply "plotlist" "()" 1))
              ("Scatter plot" (giac-apply "scatterplot" "()" 1))
              ("Stem plot" (giac-apply "batons" "()" 1))
              ("Box plot" (giac-apply "boxwhisker" "()" 1))
              ("Bar chart" (giac-apply "bar_plot" "()" 1))
              ("Pie chart" (giac-apply "camembert" "()" 1))
              ("Histogram" (giac-apply "histogram" "()" 1))))
      (-> "Probability distributions"
          (-> "Density functions"
              ("Uniform" (giac-apply "uniformd" "()" 3))
              ("Binomial" (giac-apply "binomial" "()" 3))
              ("Negative binomial" (giac-apply "negbinomial" "()" 3))
              ("Multinomial" (giac-apply "multinomial" "()" 3))
              ("Poisson" (giac-apply "poisson" "()" 2))
              ("Normal" (giac-apply "normald" "()" 3))
              ("Student" (giac-apply "studentd" "()" 2))
              ("Chi-squared" (giac-apply "chisquare" "()" 2))
              ("Fisher-Snédécor" (giac-apply "fisherd" "()" 3))
              ("Gamma" (giac-apply "gammad" "()" 3))
              ("Beta" (giac-apply "betad" "()" 3))
              ("Geometric" (giac-apply "geometric" "()" 2))
              ("Cauchy" (giac-apply "cauhcyd" "()" 3))
              ("Exponential" (giac-apply "exponentiald" "()" 2))
              ("Weibull" (giac-apply "weibulld" "()" 3))
              ("Kolmogorov-Smirnov" (giac-apply "kolmogorovd" "()" 1)))
          ("Moment generating function" (giac-apply "mgf" "()" 1))
          ("Cumulative distribution function" (giac-apply "cdf" "()" 1))
          ("Inverse distribution function" (giac-apply "icdf" "()" 1))
          ("Kernel density estimation" (giac-apply "kde" "()" 1))
          ("Distribution fitting" (giac-apply "fitdistr" "()" 2)))
      (-> "Random numbers"
          ("Random variable" (giac-apply "randvar" "()" 1))
          ("Sampling random numbers" (giac-apply "sample" "()" 1))
          ("Initializing random number generator" (giac-apply "srand" "()" 1))
          ("Random vectors" (giac-apply "randvector" "()" 1))
          ("Random matrices" (giac-apply "randmatrix" "()" 1)))
      (-> "Hypothesis testing"
          ("Z-test mean testing" (giac-apply "normalt" "()" 3))
          ("T-test mean testing" (giac-apply "studentt" "()" 3))
          ("Chi-squared distribution testing" (giac-apply "chisquaret" "()" 1))
          ("Kolmogorov-Smirnov distribution testing" (giac-apply "kolmogorovt" "()" 2))))
  (-> "Graph theory"
      (-> "Constructing graphs"
          (-> "General graphs"
              ("Undirected graph" (giac-apply "graph" "()" 1))
              ("Directed graph" (giac-apply "digraph" "()" 1)))
          (-> "Cycle and path graphs"
              ("Cycle graph" (giac-apply "cycle_graph" "()" 1))
              ("Path graph" (giac-apply "path_graph" "()" 1))
              ("Trail" (giac-apply "trail" "()" 1)))
          (-> "Complete graphs"
              ("Complete (multipartite) graph" (giac-apply "complete_graph" "()" 1))
              ("Complete binary tree" (giac-apply "complete_binary_tree" "()" 1))
              ("Complete k-ary tree" (giac-apply "complete_kary_tree" "()" 2)))
          (-> "Sequence graphs"
              ("Sequence to graph" (giac-apply "sequence_graph" "()" 1))
              ("Test for graphic sequence" (giac-apply "is_graphic_sequence" "()" 1)))
          (-> "Intersection graphs"
              ("Interval graph" (giac-apply "interval_graph" "()" 1))
              ("Kneser graph" (giac-apply "kneser_graph" "()" 2)))
          (-> "Special graphs"
              ("Hypercube graph" (giac-apply "hypercube_graph" "()" 1))
              ("Star graph" (giac-apply "star_graph" "()" 1))
              ("Wheel graph" (giac-apply "wheel_graph" "()" 1))
              ("Web graph" (giac-apply "web_graph" "()" 2))
              ("Prism graph" (giac-apply "prism_graph" "()" 1))
              ("Antiprism_graph" (giac-apply "antiprism_graph" "()" 1))
              ("Grid graph" (giac-apply "grid_graph" "()" 2))
              ("Sierpiński graph" (giac-apply "sierpinski_graph" "()" 2))
              ("Generalized Petersen graph" (giac-apply "petersen_graph" "()" 2))
              ("LCF graph" (giac-apply "lcf_graph" "()" 1)))
          (-> "Isomorphic copies"
              ("Isomorphic copy from permutation" (giac-apply "isomorphic_copy" "()" 2))
              ("Permute vertices" (giac-apply "permute_vertices" "()" 2))
              ("Relabel vertices" (giac-apply "relabel_vertices" "()" 2)))
          (-> "Subgraphs"
              ("Extract subgraph" (giac-apply "subgraph" "()" 2))
              ("Induced subgraph" (giac-apply "induced_subgraph" "()" 2))
              ("Underlying graph" (giac-apply "underlying_graph" "()" 1))
              ("Fundamental cycles" (giac-apply "cycle_basis" "()" 1))
              ("Find cycles in digraph" (giac-apply "find_cycles" "()" 1)))
          (-> "Operations on graphs"
              ("Graph complement" (giac-apply "graph_complement" "()" 1))
              ("Seidel switch" (giac-apply "seidel_switch" "()" 1))
              ("Transpose graph" (giac-apply "reverse_graph" "()" 1))
              ("Graph union" (giac-apply "graph_union" "()" 2))
              ("Disjoint union" (giac-apply "disjoint_union" "()" 2))
              ("Join two graphs" (giac-apply "graph_join" "()" 2))
              ("Power graph" (giac-apply "graph_power" "()" 2))
              ("Cartesian product" (giac-apply "cartesian_product" "()" 2))
              ("Tensor product" (giac-apply "tensor_product" "()" 2))
              ("Transitive closure" (giac-apply "transitive_closure" "()" 1))
              ("Line graph" (giac-apply "line_graph" "()" 1))
              ("Plane dual" (giac-apply "plane_dual" "()" 1))
              ("Truncate" (giac-apply "truncate_graph" "()" 1)))
          (-> "Random graphs"
              ("Random graph" (giac-apply "random_graph" "()" 1))
              ("Random digraph" (giac-apply "random_digraph" "()" 1))
              ("Random tree" (giac-apply "random_tree" "()" 1))
              ("Random planar graph" (giac-apply "random_planar_graph" "()" 1))
              ("Random graph from sequence" (giac-apply "random_sequence_graph" "()" 1))
              ("Random regular graph" (giac-apply "random_regular_graph" "()" 2))
              ("Random tournament" (giac-apply "random_tournament" "()" 1))
              ("Random network" (giac-apply "random_network" "()" 2))
              ("Randomize edge weights" (giac-apply "assign_edge_weights" "()" 2))))
      (-> "Modifying graphs"
          (-> "Promoting to directed/weighted graphs"
              ("Convert edges to arcs" (giac-apply "make_directed" "()" 2))
              ("Assign weights to edges" (giac-apply "make_weighted" "()" 2)))
          (-> "Modifying vertices"
              ("Add vertex" (giac-apply "add_vertex" "()" 2))
              ("Remove vertex" (giac-apply "delete_vertex" "()" 2)))
          (-> "Modifying edges"
              ("Add edge" (giac-apply "add_edge" "()" 2))
              ("Remove edge" (giac-apply "delete_edge" "()" 2))
              ("Add arc" (giac-apply "add_arc" "()" 2))
              ("Remove arc" (giac-apply "delete_arc" "()" 2))
              ("Set edge weight" (giac-apply "set_edge_weight" "()" 3))
              ("Edge weight" (giac-apply "get_edge_weight" "()" 2))
              ("Contract edge" (giac-apply "contract_edge" "()" 2))
              ("Subdivide edges" (giac-apply "subdivide_edges" "()" 2)))
          (-> "Attributes"
              ("Graph attribute" (giac-apply "get_graph_attribute" "()" 2))
              ("List graph attributes" (giac-apply "list_graph_attributes" "()" 1))
              ("Set graph attribute" (giac-apply "set_graph_attribute" "()" 3))
              ---
              ("Vertex attribute" (giac-apply "get_vertex_attribute" "()" 3))
              ("List vertex attributes" (giac-apply "list_vertex_attributes" "()" 2))
              ("Set vertex attribute" (giac-apply "set_vertex_attribute" "()" 4))
              ---
              ("Edge attribute" (giac-apply "get_edge_attribute" "()" 3))
              ("List edge attributes" (giac-apply "list_edge_attributes" "()" 2))
              ("Set edge attribute" (giac-apply "set_edge_attribute" "()" 4))))
      (-> "Import & export"
          ("Load graph from dot file" (giac-apply "import_graph" "()" 1))
          ("Write graph to dot file" (giac-apply "export_graph" "()" 2)))
      (-> "Graph properties"
          (-> "Basic"
              ("Vertex degree" (giac-apply "vertex_degree" "()" 2))
              ("Get edges incident to vertex" (giac-apply "incident_edges" "()" 2)))
          (-> "Algebraic"
              ("Adjacency matrix" (giac-apply "adjacency_matrix" "()" 1))
              ("Weight matrix" (giac-apply "weight_matrix" "()" 1))
              ("Incidence matrix" (giac-apply "incidence_matrix" "()" 1))
              ("Laplacian matrix" (giac-apply "laplacian_matrix" "()" 1))
              ("Characteristic polynomial" (giac-apply "graph_charpoly" "()" 1))
              ("Graph spectrum" (giac-apply "graph_spectrum" "()" 1))
              ("Seidel spectrum" (giac-apply "seidel_spectrum" "()" 1)))
          (-> "Isomorphism"
              ("Test graph isomorphism" (giac-apply "is_isomorphic" "()" 2))
              ("Test subgraph isomorphism" (giac-apply "is_subgraph_isomorphic" "()" 1))
              ("Canonical labeling" (giac-apply "canonical_labeling" "()" 1))
              ("Automorphisms" (giac-apply "graph_automorphisms" "()" 1)))
          (-> "Polynomials"
              ("Tutte polynomial" (giac-apply "tutte_polynomial" "()" 1))
              ("Chromatic polynomial" (giac-apply "chromatic_polynomial" "()" 1))
              ("Flow polynomial" (giac-apply "flow_polynomial" "()" 1))
              ("Reliability polynomial" (giac-apply "reliability_polynomial" "()" 1)))
          (-> "Connectivity"
              ("Vertex connectivity" (giac-apply "vertex_connectivity" "()" 1))
              ("Edge connectivity" (giac-apply "edge_connectivity" "()" 1))
              ("Graph rank" (giac-apply "graph_rank" "()" 1))
              ("Articulation points" (giac-apply "articulation_points" "()" 1))
              ("Connected components" (giac-apply "connected_components" "()" 1))
              ("Strongly connected components" (giac-apply "strongly_connected_components" "()" 1))
              ("Biconnected components" (giac-apply "biconnected_components" "()" 1)))
          (-> "Trees"
              ("Tree height" (giac-apply "tree_height" "()" 1))
              ("Lowest common ancestor" (giac-apply "line_graph" "()" 3)))
          (-> "Networks"
              ("Maximum flow" (giac-apply "maxflow" "()" 3))
              ("Minimum cut" (giac-apply "minimum_cut" "()" 3))
              ("Clustering coefficient" (giac-apply "clustering_coefficient" "()" 1))
              ("Network transitivity" (giac-apply "network_transitivity" "()" 1)))
          (-> "Distance in graphs"
              ("Vertex distance" (giac-apply "vertex_distance" "()" 3))
              ("All-pairs vertex distance" (giac-apply "allpairs_distance" "()" 1))
              ("Graph diameter" (giac-apply "graph_diameter" "()" 1))
              ("Girth" (giac-apply "girth" "()" 1)))
          (-> "Acyclic graphs"
              ("Topologic sorting" (giac-apply "topologic_sort" "()" 1))
              ("st-ordering" (giac-apply "st_ordering" "()" 3))
              ("Graph condensation" (giac-apply "condensation" "()" 1)))
          (-> "Matching"
              ("Maximum matching" (giac-apply "maximum_matching" "()" 1))
              ("Maximum matching in bipartite graph" (giac-apply "bipartite_matching" "()" 1)))
          (-> "Cliques"
              ("Maximal cliques" (giac-apply "find_cliques" "()" 1))
              ("Maximum clique" (giac-apply "maximum_clique" "()" 1))
              ("Maximum independent set" (giac-apply "maximum_independent_set" "()" 1))
              ("Clique cover" (giac-apply "clique_cover" "()" 1))
              ("Clique cover number" (giac-apply "clique_cover_number" "()" 1)))
          (-> "Coloring"
              ("Greedy vertex coloring" (giac-apply "greedy_color" "()" 1))
              ("Minimal vertex coloring" (giac-apply "minimal_vertex_coloring" "()" 1))
              ("Chromatic number" (giac-apply "chromatic_number" "()" 1))
              ("Minimal edge coloring" (giac-apply "minimal_edge_coloring" "()" 1))))
      (-> "Traversing graphs"
          (-> "Optimal routing"
              ("Shortest path in unweighted graph" (giac-apply "shortest_path" "()" 3))
              ("Cheapest path in weighted graph" (giac-apply "dijkstra" "()" 3))
              ("k-shortest paths" (giac-apply "kspaths" "()" 4))
              ("Shortest tour" (giac-apply "traveling_salesman" "()" 1)))
          (-> "Spanning trees"
              ("Spanning tree" (giac-apply "spanning_tree" "()" 1))
              ("Number of spanning trees" (giac-apply "number_of_spanning_trees" "()" 1))
              ("Minimal spanning tree" (giac-apply "minimal_spanning_tree" "()" 1))))
      (-> "Visualizing graphs"
          ("Draw graph" (giac-apply "draw_graph" "()" 1))
          ("Set vertex positions" (giac-apply "set_vertex_positions" "()" 2))
          ("Highlight vertex" (giac-apply "highlight_vertex" "()" 2))
          ("Highlight edges" (giac-apply "highlight_edges" "()" 2))
          ("Highlight trail" (giac-apply "highlight_trail" "()" 2))
          ("Highlight subgraph" (giac-apply "highlight_subgraph" "()" 2))))
  (-> "Optimization"
      ("Minimize" (giac-apply "minimize" "()" 2))
      ("Maximize" (giac-apply "maximize" "()" 2))
      ("Solve linear programming problem" (giac-apply "lpsolve" "()" 2))
      ("Solve nonlinear programming problem" (giac-apply "nlpsolve" "()" 2))
      ("Solve transportation problem" (giac-apply "tpsolve" "()" 3)))
  (-> "Signal processing"
      ("Root mean square (RMS)" (giac-apply "rms" "()" 1))
      ("Cross-correlation" (giac-apply "cross_correlation" "()" 2))
      ("Auto-correlation" (giac-apply "auto_correlation" "()" 1))
      ("Convolution" (giac-apply "convolution" "()" 2))
      ("Clamp range" (giac-apply "threshold" "()" 3))
      (-> "Filter"
          ("Low-pass" (giac-apply "lowpass" "()" 2))
          ("High-pass" (giac-apply "highpass" "()" 2))
          ("Moving average" (giac-apply "moving_average" "()" 2)))
      (-> "Common functions"
          ("Rectangular function" (giac-apply "rect" "()" 1))
          ("Boxcar function" (giac-apply "boxcar" "()" 3))
          ("Triangular function" (giac-apply "tri" "()" 1))
          ("Sampling function" (giac-apply "sinc" "()" 1))
          ("Step function" (giac-apply "Heaviside" "()" 1))
          ("Dirac delta function" (giac-apply "Dirac" "()" 1)))
      (-> "Window functions"
          ("Bartlett-Hann" (giac-apply "bartlett_hann_window" "()" 1))
          ("Blackman" (giac-apply "blackman_window" "()" 1))
          ("Blackman-Harris" (giac-apply "blackman_harris_window" "()" 1))
          ("Bohman" (giac-apply "bohman_window" "()" 1))
          ("Cosine" (giac-apply "cosine_window" "()" 1))
          ("Gaussian" (giac-apply "gaussian_window" "()" 1))
          ("Hamming" (giac-apply "hamming_window" "()" 1))
          ("Hann" (giac-apply "hann_window" "()" 1))
          ("Hann-Poisson" (giac-apply "hann_poisson_window" "()" 1))
          ("Parzen" (giac-apply "parzen_window" "()" 1))
          ("Poisson" (giac-apply "poisson_window" "()" 1))
          ("Riemann" (giac-apply "riemann_window" "()" 1))
          ("Triangular" (giac-apply "triangle_window" "()" 1))
          ("Tukey" (giac-apply "tukey_window" "()" 1))
          ("Welch" (giac-apply "welch_window" "()" 1)))
      (-> "Transformations"
          ("Fourier cosine coefficient" (giac-apply "fourier_an" "()" 1))
          ("Fourier sine coefficient" (giac-apply "fourier_bn" "()" 1))
          ("Fourier coefficient" (giac-apply "fourier_cn" "()" 1))
          ("Laplace transform" (giac-apply "laplace" "()" 1))
          ("Inverse Laplace transform" (giac-apply "ilaplace" "()" 1))
          ("Fourier transform" (giac-apply "fourier" "()" 1))
          ("Inverse Fourier transform" (giac-apply "ifourier" "()" 1))
          ("Fast Fourier Transform (FFT)" (giac-apply "fft" "()" 1))
          ("Inverse FFT" (giac-apply "ifft" "()" 1))
          ("Z-transform" (giac-apply "ztrans" "()" 1)))
      (-> "Audio tools"
          ("Create audio clip" (giac-apply "createwav" "()" 1))
          ("Read audio from file" (giac-apply "readwav" "()" 1))
          ("Write audio to file" (giac-apply "writewav" "()" 2))
          ("Stereo to mono" (giac-apply "stereo2mono" "()" 1))
          ("Resample" (giac-apply "resample" "()" 1))
          ("Show waveform" (giac-apply "plotwav" "()" 1))
          ("Show spectrum" (giac-apply "plotspectrum" "()" 1))))

  (if (not-in-session?)
      ---
      (link scripts-eval-toggle-menu)))

(menu-bind plugin-menu
  (:require (or (in-giac?) (and (not-in-session?) (giac-scripts?))))
  (=> "Giac" (link giac-functions-menu)))
